const User = require('../models/User');
const jwt = require('jsonwebtoken');
const axios = require('axios');
const config = require('../config/config');
const facebookService = require('../services/facebookService');
const instagramService = require('../services/instagramService');

/**
 * Register a new user
 */
exports.register = async (req, res) => {
  try {
    const { email, password, name } = req.body;

    // Check if user already exists
    const existingUser = await User.findOne({ email });
    if (existingUser) {
      return res.status(400).json({ error: 'Email already registered' });
    }

    // Create new user
    const user = new User({ email, password, name });
    await user.save();

    // Generate JWT token
    const token = jwt.sign({ userId: user._id }, config.jwtSecret, {
      expiresIn: '30d',
    });

    res.status(201).json({
      message: 'User registered successfully',
      token,
      user: {
        id: user._id,
        email: user.email,
        name: user.name,
      },
    });
  } catch (error) {
    console.error('Register error:', error);
    res.status(500).json({ error: 'Registration failed' });
  }
};

/**
 * Login user
 */
exports.login = async (req, res) => {
  try {
    const { email, password } = req.body;

    // Find user
    const user = await User.findOne({ email });
    if (!user) {
      return res.status(401).json({ error: 'Invalid credentials' });
    }

    // Check password
    const isMatch = await user.comparePassword(password);
    if (!isMatch) {
      return res.status(401).json({ error: 'Invalid credentials' });
    }

    // Generate JWT token
    const token = jwt.sign({ userId: user._id }, config.jwtSecret, {
      expiresIn: '30d',
    });

    res.json({
      token,
      user: {
        id: user._id,
        email: user.email,
        name: user.name,
        connectedAccounts: user.connectedAccounts,
      },
    });
  } catch (error) {
    console.error('Login error:', error);
    res.status(500).json({ error: 'Login failed' });
  }
};

/**
 * Get Facebook login URL
 */
exports.getFacebookAuthUrl = (req, res) => {
  const redirectUri = `${req.protocol}://${req.get('host')}/api/auth/facebook/callback`;
  const scope = 'pages_messaging,pages_manage_metadata,instagram_basic,instagram_manage_messages,pages_show_list';
  
  const authUrl = `https://www.facebook.com/v18.0/dialog/oauth?` +
    `client_id=${config.facebook.appId}` +
    `&redirect_uri=${encodeURIComponent(redirectUri)}` +
    `&scope=${encodeURIComponent(scope)}` +
    `&state=${req.userId}`; // Pass user ID in state

  res.json({ authUrl });
};

/**
 * Facebook OAuth callback
 */
exports.facebookCallback = async (req, res) => {
  try {
    const { code, state: userId } = req.query;

    if (!code) {
      return res.redirect(`${config.frontend.url}/connect?error=no_code`);
    }

    const redirectUri = `${req.protocol}://${req.get('host')}/api/auth/facebook/callback`;

    // Exchange code for access token
    const tokenResponse = await axios.get(
      `https://graph.facebook.com/v18.0/oauth/access_token`,
      {
        params: {
          client_id: config.facebook.appId,
          client_secret: config.facebook.appSecret,
          redirect_uri: redirectUri,
          code,
        },
      }
    );

    const userAccessToken = tokenResponse.data.access_token;

    // Get long-lived token
    const longLivedToken = await facebookService.getLongLivedToken(userAccessToken);

    // Get user's Facebook pages
    const pages = await facebookService.getUserPages(longLivedToken);

    if (pages && pages.length > 0) {
      const page = pages[0]; // Use first page (can be extended to select multiple)

      // Get Instagram account if linked
      let instagramAccount = null;
      try {
        instagramAccount = await instagramService.getInstagramAccount(page.access_token, page.id);
      } catch (error) {
        console.log('No Instagram account linked to this page');
      }

      // Update user with connected accounts
      const user = await User.findById(userId);
      if (user) {
        user.connectedAccounts.facebook = {
          pageId: page.id,
          pageName: page.name,
          accessToken: page.access_token,
          isConnected: true,
        };

        if (instagramAccount) {
          const igProfile = await instagramService.getProfile(page.access_token, instagramAccount.id);
          user.connectedAccounts.instagram = {
            accountId: instagramAccount.id,
            username: igProfile?.username,
            accessToken: page.access_token,
            isConnected: true,
          };
        }

        await user.save();
      }
    }

    res.redirect(`${config.frontend.url}/connect?success=true`);
  } catch (error) {
    console.error('Facebook callback error:', error.response?.data || error.message);
    res.redirect(`${config.frontend.url}/connect?error=auth_failed`);
  }
};

/**
 * Connect WhatsApp Business Account
 */
exports.connectWhatsApp = async (req, res) => {
  try {
    const { phoneNumberId, accessToken, displayPhoneNumber } = req.body;
    const userId = req.userId;

    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    user.connectedAccounts.whatsapp = {
      phoneNumberId,
      displayPhoneNumber,
      accessToken,
      isConnected: true,
    };

    await user.save();

    res.json({
      message: 'WhatsApp connected successfully',
      connectedAccounts: user.connectedAccounts,
    });
  } catch (error) {
    console.error('Connect WhatsApp error:', error);
    res.status(500).json({ error: 'Failed to connect WhatsApp' });
  }
};

/**
 * Get current user info
 */
exports.getCurrentUser = async (req, res) => {
  try {
    const user = await User.findById(req.userId).select('-password');
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    res.json({ user });
  } catch (error) {
    console.error('Get current user error:', error);
    res.status(500).json({ error: 'Failed to get user info' });
  }
};
