const User = require('../models/User');
const Conversation = require('../models/Conversation');
const Message = require('../models/Message');
const facebookService = require('../services/facebookService');
const instagramService = require('../services/instagramService');
const whatsappService = require('../services/whatsappService');

/**
 * Get all conversations for the authenticated user
 */
exports.getConversations = async (req, res) => {
  try {
    const userId = req.userId;
    const { platform, page = 1, limit = 20 } = req.query;

    const query = { userId };
    if (platform) {
      query.platform = platform;
    }

    const conversations = await Conversation.find(query)
      .sort({ updatedAt: -1 })
      .limit(limit)
      .skip((page - 1) * limit);

    const total = await Conversation.countDocuments(query);

    res.json({
      conversations,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit),
      },
    });
  } catch (error) {
    console.error('Get conversations error:', error);
    res.status(500).json({ error: 'Failed to fetch conversations' });
  }
};

/**
 * Get messages for a specific conversation
 */
exports.getMessages = async (req, res) => {
  try {
    const { conversationId } = req.params;
    const { page = 1, limit = 50 } = req.query;
    const userId = req.userId;

    // Verify conversation belongs to user
    const conversation = await Conversation.findOne({
      _id: conversationId,
      userId,
    });

    if (!conversation) {
      return res.status(404).json({ error: 'Conversation not found' });
    }

    const messages = await Message.find({ conversationId })
      .sort({ timestamp: -1 })
      .limit(limit)
      .skip((page - 1) * limit);

    const total = await Message.countDocuments({ conversationId });

    // Mark conversation as read
    conversation.unreadCount = 0;
    await conversation.save();

    res.json({
      messages: messages.reverse(), // Return in chronological order
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit),
      },
    });
  } catch (error) {
    console.error('Get messages error:', error);
    res.status(500).json({ error: 'Failed to fetch messages' });
  }
};

/**
 * Send a message
 */
exports.sendMessage = async (req, res) => {
  try {
    const { conversationId, text } = req.body;
    const userId = req.userId;

    if (!text || !conversationId) {
      return res.status(400).json({ error: 'Missing required fields' });
    }

    // Get conversation and verify ownership
    const conversation = await Conversation.findOne({
      _id: conversationId,
      userId,
    });

    if (!conversation) {
      return res.status(404).json({ error: 'Conversation not found' });
    }

    // Get user credentials
    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    const platform = conversation.platform;
    const recipientId = conversation.platformConversationId;

    let platformResponse;

    // Send message based on platform
    switch (platform) {
      case 'facebook': {
        const accessToken = user.connectedAccounts.facebook.accessToken;
        platformResponse = await facebookService.sendMessage(
          accessToken,
          recipientId,
          text
        );
        break;
      }

      case 'instagram': {
        const accessToken = user.connectedAccounts.instagram.accessToken;
        platformResponse = await instagramService.sendMessage(
          accessToken,
          recipientId,
          text
        );
        break;
      }

      case 'whatsapp': {
        const phoneNumberId = user.connectedAccounts.whatsapp.phoneNumberId;
        const accessToken = user.connectedAccounts.whatsapp.accessToken;
        platformResponse = await whatsappService.sendMessage(
          phoneNumberId,
          accessToken,
          recipientId,
          text
        );
        break;
      }

      default:
        return res.status(400).json({ error: 'Invalid platform' });
    }

    // Save message to database
    const message = new Message({
      conversationId: conversation._id,
      platform,
      platformMessageId: platformResponse.message_id || platformResponse.messages?.[0]?.id,
      senderId: user._id.toString(),
      senderName: user.name,
      text,
      isFromBusiness: true,
      status: 'sent',
      timestamp: new Date(),
    });
    await message.save();

    // Update conversation
    conversation.lastMessage = {
      text,
      timestamp: new Date(),
      isFromUser: true,
    };
    conversation.updatedAt = new Date();
    await conversation.save();

    res.json({
      message: 'Message sent successfully',
      data: message,
    });
  } catch (error) {
    console.error('Send message error:', error);
    res.status(500).json({ error: 'Failed to send message' });
  }
};

/**
 * Get conversation statistics
 */
exports.getStats = async (req, res) => {
  try {
    const userId = req.userId;

    const totalConversations = await Conversation.countDocuments({ userId });
    const unreadConversations = await Conversation.countDocuments({
      userId,
      unreadCount: { $gt: 0 },
    });

    const platformStats = await Conversation.aggregate([
      { $match: { userId: mongoose.Types.ObjectId(userId) } },
      {
        $group: {
          _id: '$platform',
          count: { $sum: 1 },
          unread: {
            $sum: {
              $cond: [{ $gt: ['$unreadCount', 0] }, 1, 0],
            },
          },
        },
      },
    ]);

    res.json({
      total: totalConversations,
      unread: unreadConversations,
      byPlatform: platformStats,
    });
  } catch (error) {
    console.error('Get stats error:', error);
    res.status(500).json({ error: 'Failed to fetch statistics' });
  }
};
