const User = require('../models/User');
const Conversation = require('../models/Conversation');
const Message = require('../models/Message');
const config = require('../config/config');
const facebookService = require('../services/facebookService');
const instagramService = require('../services/instagramService');
const whatsappService = require('../services/whatsappService');

/**
 * Webhook verification for Facebook/Instagram
 */
exports.verifyFacebookWebhook = (req, res) => {
  const mode = req.query['hub.mode'];
  const token = req.query['hub.verify_token'];
  const challenge = req.query['hub.challenge'];

  if (mode === 'subscribe' && token === config.webhook.verifyToken) {
    console.log('✅ Facebook webhook verified');
    res.status(200).send(challenge);
  } else {
    console.error('❌ Facebook webhook verification failed');
    res.sendStatus(403);
  }
};

/**
 * Handle incoming Facebook/Instagram messages
 */
exports.handleFacebookWebhook = async (req, res) => {
  try {
    const body = req.body;

    if (body.object === 'page') {
      for (const entry of body.entry) {
        const webhookEvent = entry.messaging?.[0] || entry.changes?.[0];

        if (!webhookEvent) continue;

        // Determine platform (Facebook Messenger or Instagram)
        const platform = entry.messaging ? 'facebook' : 'instagram';

        // Handle message event
        if (webhookEvent.message) {
          await handleIncomingMessage(webhookEvent, platform, entry.id);
        }
      }
    }

    res.status(200).send('EVENT_RECEIVED');
  } catch (error) {
    console.error('Facebook webhook error:', error);
    res.sendStatus(500);
  }
};

/**
 * Webhook verification for WhatsApp
 */
exports.verifyWhatsAppWebhook = (req, res) => {
  const mode = req.query['hub.mode'];
  const token = req.query['hub.verify_token'];
  const challenge = req.query['hub.challenge'];

  if (mode === 'subscribe' && token === config.webhook.verifyToken) {
    console.log('✅ WhatsApp webhook verified');
    res.status(200).send(challenge);
  } else {
    console.error('❌ WhatsApp webhook verification failed');
    res.sendStatus(403);
  }
};

/**
 * Handle incoming WhatsApp messages
 */
exports.handleWhatsAppWebhook = async (req, res) => {
  try {
    const body = req.body;

    if (body.object === 'whatsapp_business_account') {
      for (const entry of body.entry) {
        for (const change of entry.changes) {
          if (change.value.messages) {
            for (const message of change.value.messages) {
              await handleWhatsAppMessage(message, change.value);
            }
          }

          // Handle message status updates
          if (change.value.statuses) {
            for (const status of change.value.statuses) {
              await handleWhatsAppStatus(status);
            }
          }
        }
      }
    }

    res.status(200).send('EVENT_RECEIVED');
  } catch (error) {
    console.error('WhatsApp webhook error:', error);
    res.sendStatus(500);
  }
};

/**
 * Handle incoming message from Facebook/Instagram
 */
async function handleIncomingMessage(webhookEvent, platform, pageId) {
  const senderId = webhookEvent.sender.id;
  const messageText = webhookEvent.message.text;
  const messageId = webhookEvent.message.mid;
  const timestamp = webhookEvent.timestamp;

  // Find user by page ID
  const user = await User.findOne({
    $or: [
      { 'connectedAccounts.facebook.pageId': pageId },
      { 'connectedAccounts.instagram.accountId': pageId },
    ],
  });

  if (!user) {
    console.log(`No user found for ${platform} page:`, pageId);
    return;
  }

  // Get or create conversation
  let conversation = await Conversation.findOne({
    userId: user._id,
    platform,
    platformConversationId: senderId,
  });

  if (!conversation) {
    // Get sender profile
    const accessToken = platform === 'facebook'
      ? user.connectedAccounts.facebook.accessToken
      : user.connectedAccounts.instagram.accessToken;

    const profile = platform === 'facebook'
      ? await facebookService.getUserProfile(accessToken, senderId)
      : await instagramService.getProfile(accessToken, senderId);

    conversation = new Conversation({
      userId: user._id,
      platform,
      platformConversationId: senderId,
      participant: {
        id: senderId,
        name: profile?.first_name 
          ? `${profile.first_name} ${profile.last_name || ''}`
          : profile?.username || 'Unknown',
        profilePic: profile?.profile_pic || profile?.profile_picture_url,
      },
    });
  }

  // Update conversation
  conversation.lastMessage = {
    text: messageText,
    timestamp: new Date(timestamp),
    isFromUser: false,
  };
  conversation.unreadCount += 1;
  conversation.updatedAt = new Date();
  await conversation.save();

  // Save message
  const message = new Message({
    conversationId: conversation._id,
    platform,
    platformMessageId: messageId,
    senderId,
    senderName: conversation.participant.name,
    text: messageText,
    isFromBusiness: false,
    timestamp: new Date(timestamp),
  });
  await message.save();

  console.log(`✅ Received ${platform} message from ${conversation.participant.name}`);
}

/**
 * Handle incoming WhatsApp message
 */
async function handleWhatsAppMessage(message, value) {
  const senderId = message.from;
  const messageText = message.text?.body;
  const messageId = message.id;
  const timestamp = message.timestamp;

  // Find user by WhatsApp phone number ID
  const user = await User.findOne({
    'connectedAccounts.whatsapp.phoneNumberId': value.metadata.phone_number_id,
  });

  if (!user) {
    console.log('No user found for WhatsApp phone number:', value.metadata.phone_number_id);
    return;
  }

  // Get or create conversation
  let conversation = await Conversation.findOne({
    userId: user._id,
    platform: 'whatsapp',
    platformConversationId: senderId,
  });

  if (!conversation) {
    const senderName = value.contacts?.[0]?.profile?.name || senderId;
    
    conversation = new Conversation({
      userId: user._id,
      platform: 'whatsapp',
      platformConversationId: senderId,
      participant: {
        id: senderId,
        name: senderName,
      },
    });
  }

  // Update conversation
  conversation.lastMessage = {
    text: messageText,
    timestamp: new Date(parseInt(timestamp) * 1000),
    isFromUser: false,
  };
  conversation.unreadCount += 1;
  conversation.updatedAt = new Date();
  await conversation.save();

  // Save message
  const msg = new Message({
    conversationId: conversation._id,
    platform: 'whatsapp',
    platformMessageId: messageId,
    senderId,
    senderName: conversation.participant.name,
    text: messageText,
    isFromBusiness: false,
    timestamp: new Date(parseInt(timestamp) * 1000),
  });
  await msg.save();

  // Mark as read
  await whatsappService.markAsRead(
    value.metadata.phone_number_id,
    user.connectedAccounts.whatsapp.accessToken,
    messageId
  );

  console.log(`✅ Received WhatsApp message from ${conversation.participant.name}`);
}

/**
 * Handle WhatsApp message status updates
 */
async function handleWhatsAppStatus(status) {
  const messageId = status.id;
  const newStatus = status.status; // sent, delivered, read, failed

  await Message.findOneAndUpdate(
    { platformMessageId: messageId },
    { status: newStatus }
  );
}
