const axios = require('axios');
const config = require('../config/config');

class FacebookService {
  constructor() {
    this.baseUrl = `https://graph.facebook.com/${config.facebook.graphApiVersion}`;
  }

  /**
   * Send a message via Facebook Messenger
   */
  async sendMessage(pageAccessToken, recipientId, messageText) {
    try {
      const response = await axios.post(
        `${this.baseUrl}/me/messages`,
        {
          recipient: { id: recipientId },
          message: { text: messageText },
        },
        {
          params: { access_token: pageAccessToken },
        }
      );
      return response.data;
    } catch (error) {
      console.error('Facebook send message error:', error.response?.data || error.message);
      throw error;
    }
  }

  /**
   * Mark message as read
   */
  async markAsRead(pageAccessToken, senderId) {
    try {
      await axios.post(
        `${this.baseUrl}/me/messages`,
        {
          recipient: { id: senderId },
          sender_action: 'mark_seen',
        },
        {
          params: { access_token: pageAccessToken },
        }
      );
    } catch (error) {
      console.error('Facebook mark as read error:', error.response?.data || error.message);
    }
  }

  /**
   * Get user profile information
   */
  async getUserProfile(pageAccessToken, userId) {
    try {
      const response = await axios.get(`${this.baseUrl}/${userId}`, {
        params: {
          fields: 'first_name,last_name,profile_pic',
          access_token: pageAccessToken,
        },
      });
      return response.data;
    } catch (error) {
      console.error('Facebook get user profile error:', error.response?.data || error.message);
      return null;
    }
  }

  /**
   * Get long-lived page access token
   */
  async getLongLivedToken(shortLivedToken) {
    try {
      const response = await axios.get(`${this.baseUrl}/oauth/access_token`, {
        params: {
          grant_type: 'fb_exchange_token',
          client_id: config.facebook.appId,
          client_secret: config.facebook.appSecret,
          fb_exchange_token: shortLivedToken,
        },
      });
      return response.data.access_token;
    } catch (error) {
      console.error('Facebook get long-lived token error:', error.response?.data || error.message);
      throw error;
    }
  }

  /**
   * Get user's Facebook pages
   */
  async getUserPages(userAccessToken) {
    try {
      const response = await axios.get(`${this.baseUrl}/me/accounts`, {
        params: {
          access_token: userAccessToken,
        },
      });
      return response.data.data;
    } catch (error) {
      console.error('Facebook get pages error:', error.response?.data || error.message);
      throw error;
    }
  }
}

module.exports = new FacebookService();
