const axios = require('axios');
const config = require('../config/config');

class WhatsAppService {
  constructor() {
    this.baseUrl = `https://graph.facebook.com/${config.facebook.graphApiVersion}`;
  }

  /**
   * Send a text message via WhatsApp Business API
   */
  async sendMessage(phoneNumberId, accessToken, recipientPhone, messageText) {
    try {
      const response = await axios.post(
        `${this.baseUrl}/${phoneNumberId}/messages`,
        {
          messaging_product: 'whatsapp',
          to: recipientPhone,
          type: 'text',
          text: { body: messageText },
        },
        {
          headers: {
            Authorization: `Bearer ${accessToken}`,
            'Content-Type': 'application/json',
          },
        }
      );
      return response.data;
    } catch (error) {
      console.error('WhatsApp send message error:', error.response?.data || error.message);
      throw error;
    }
  }

  /**
   * Mark message as read
   */
  async markAsRead(phoneNumberId, accessToken, messageId) {
    try {
      await axios.post(
        `${this.baseUrl}/${phoneNumberId}/messages`,
        {
          messaging_product: 'whatsapp',
          status: 'read',
          message_id: messageId,
        },
        {
          headers: {
            Authorization: `Bearer ${accessToken}`,
            'Content-Type': 'application/json',
          },
        }
      );
    } catch (error) {
      console.error('WhatsApp mark as read error:', error.response?.data || error.message);
    }
  }

  /**
   * Send a template message (for initial outreach after 24h window)
   */
  async sendTemplateMessage(phoneNumberId, accessToken, recipientPhone, templateName, languageCode = 'en') {
    try {
      const response = await axios.post(
        `${this.baseUrl}/${phoneNumberId}/messages`,
        {
          messaging_product: 'whatsapp',
          to: recipientPhone,
          type: 'template',
          template: {
            name: templateName,
            language: { code: languageCode },
          },
        },
        {
          headers: {
            Authorization: `Bearer ${accessToken}`,
            'Content-Type': 'application/json',
          },
        }
      );
      return response.data;
    } catch (error) {
      console.error('WhatsApp send template error:', error.response?.data || error.message);
      throw error;
    }
  }

  /**
   * Download media file
   */
  async downloadMedia(mediaId, accessToken) {
    try {
      // First get media URL
      const mediaUrlResponse = await axios.get(`${this.baseUrl}/${mediaId}`, {
        headers: { Authorization: `Bearer ${accessToken}` },
      });
      
      const mediaUrl = mediaUrlResponse.data.url;
      
      // Download the media
      const mediaResponse = await axios.get(mediaUrl, {
        headers: { Authorization: `Bearer ${accessToken}` },
        responseType: 'arraybuffer',
      });
      
      return {
        data: mediaResponse.data,
        contentType: mediaResponse.headers['content-type'],
      };
    } catch (error) {
      console.error('WhatsApp download media error:', error.response?.data || error.message);
      throw error;
    }
  }
}

module.exports = new WhatsAppService();
